using System;
using System.Reflection;
using BLOODUNIT = gov.va.med.vbecs.Common.VbecsTables.BloodUnit;
using BLOODUNITMEDIA = gov.va.med.vbecs.Common.VbecsTables.BloodUnitMedia;
using BLOODUNITFINANCE = gov.va.med.vbecs.Common.VbecsTables.BloodUnitFinance;
using BLOODUNITSTATUS = gov.va.med.vbecs.Common.VbecsTables.BloodUnitStatus;
using BLOODUNITMODIFICATION = gov.va.med.vbecs.Common.VbecsTables.BloodUnitModification;
using SHIPMENT = gov.va.med.vbecs.Common.VbecsTables.Shipment;
using MODIFICATIONSUPPLY = gov.va.med.vbecs.Common.VbecsTables.ModificationSupply;
using MODIFICATIONEQUIPMENT = gov.va.med.vbecs.Common.VbecsTables.ModificationEquipment;
using BLOODUNITSHIPMENT = gov.va.med.vbecs.Common.VbecsTables.BloodUnitShipment;

namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// Summary description for ModifiedUnit.
	/// </summary>
	public class ModifiedUnit : BaseBusinessObject
	{

		#region Variables

		private const string M_CONTAINER = "Container";
		private	const string MIN_COLUMNS =	BLOODUNITMODIFICATION.BloodUnitModificationGuid + "," + BLOODUNITMODIFICATION.DivisionCode + "," + BLOODUNITMODIFICATION.FromBloodUnitGuid + "," + BLOODUNITMODIFICATION.ModificationDate + "," +
											BLOODUNITMODIFICATION.ModificationMethodCode + "," + BLOODUNITMODIFICATION.ProductModificationCode + "," + BLOODUNITMODIFICATION.ToBloodUnitGuid;

		/// <summary>
		/// BloodUnit being modified - not sure if I'll need this
		/// </summary>
		private BOL.BloodUnit _newBloodUnit;
		
		private System.Collections.ArrayList _equipmentList;
		private System.Collections.ArrayList _supplyList;

		private BOL.Supply _scdWafer;
		private BOL.Supply _container;

		/// <summary>
		/// Target BloodProduct - we use this to avoid the BloodUnit pitfalls when values change 
		/// </summary>
		private BOL.BloodProduct _targetBloodProduct;

		/// <summary>
		/// We save this off to assign it for the unit that chooses to use the orginal unit's container
		/// </summary>
		private string _originalContainerLotNumber;
		private bool _originalContainerIndicator = false;

		private bool _weldIndicator = false;

		private System.DateTime _originalExpirationDateTime = System.DateTime.MaxValue;
		private System.DateTime _unitExpirationDateTime;

		/// <summary>
		/// This is a datarow with columns matching the BloodUnitModification table
		/// </summary>
		private System.Data.DataRow _modifiedUnitRow;
		
		//CR2002 - Removing Open/Closed Expiration int values, variables are now public access in BloodUnitModification class

		//Exceptions
		private System.Collections.Hashtable _supplyExceptions = new System.Collections.Hashtable();
		private System.Collections.Hashtable _containerExceptions = new System.Collections.Hashtable();
		private System.Collections.Hashtable _waferExceptions = new System.Collections.Hashtable();

		private bool _labelVerificationFailed = false;

		#endregion

		#region Constructors

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="105"> 
		///		<ExpectedInput>Valid parameters</ExpectedInput>
		///		<ExpectedOutput>ModifiedUnit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="106"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor - with lots of parameters for several reasons.
		/// 1 - this way we don't have to set them one at a time
		/// 2 - this way we don't forget and need broken rules?
		/// 3 - this way the properties can be read only, and don't have to worry about changing/losing values
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="bumControl"></param>
		public ModifiedUnit(System.Guid bloodUnitGuid, BOL.BloodUnitModification bumControl)
		{
			//CR2002 - Removing Open/Closed Expiration int values, variables are now public access in the BloodUnitModification object
			if (bumControl.ProductModificationCode == Common.ProductModification.Unknown)
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument(BLOODUNITMODIFICATION.ProductModificationCode).ResString);

			if (bumControl.ModificationMethod == Common.ModifyUnitMethod.None)
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument(BLOODUNITMODIFICATION.ModificationMethodCode).ResString);

			if (bloodUnitGuid == System.Guid.Empty)
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument(BLOODUNIT.BloodUnitGuid).ResString);

			//We can get about everthing we'll need from the existing bloodunit
			this._newBloodUnit = new BOL.BloodUnit(bloodUnitGuid);
			this._modifiedUnitRow = DAL.BloodUnitModification.GetEmptyBloodUnitModificationTable(false).NewRow();

			//Assign the modification-specific values
			this._modifiedUnitRow[BLOODUNITMODIFICATION.ProductModificationCode] = Common.Utility.GetProductModificationCharFromEnum(bumControl.ProductModificationCode);
			this._modifiedUnitRow[BLOODUNITMODIFICATION.ModificationMethodCode] = Common.Utility.GetModificationMethodCharFromEnum(bumControl.ModificationMethod);
			this._modifiedUnitRow[BLOODUNITMODIFICATION.ModificationDate] = bumControl.ModificationDateTime;

			this.MakeNew(bumControl);
			this.AttachBrokenRulesEvents();
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="107"> 
		///		<ExpectedInput>Valid DataRow</ExpectedInput>
		///		<ExpectedOutput>ModifiedUnit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="108"> 
		///		<ExpectedInput>Invalid DataRow</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Populates the class with information for an already modified unit  -- used in Edit existing pool
		/// </summary>
		/// <param name="dtRow"></param>
		public ModifiedUnit(System.Data.DataRow dtRow)
		{
			if (!Common.Utility.VerifyMinColumnsInDataTable(MIN_COLUMNS.Split(','), dtRow.Table))
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InsufficientColumns(BLOODUNITMODIFICATION.TableName).ResString);

			this._modifiedUnitRow = DAL.BloodUnitModification.GetEmptyBloodUnitModificationTable(false).NewRow();

			this.LoadFromDataRow(dtRow);
			
			this._newBloodUnit = new BOL.BloodUnit(this.ToBloodUnitGuid);

			//Create our starting point
			this._targetBloodProduct = new BOL.BloodProduct(this._newBloodUnit.BloodProductGuid);

			this._equipmentList = new System.Collections.ArrayList();
			this._supplyList = new System.Collections.ArrayList();
			this._container = new BOL.Supply(System.Guid.Empty);
			this._scdWafer = new BOL.Supply(System.Guid.Empty);

			this.AttachBrokenRulesEvents();
		}

		#endregion

		#region Events

		/// <summary>
		/// Event to capture the modified unit's volume change
		/// </summary>
		public event EventHandler UnitVolumeChanged;

		/// <summary>
		/// Fires the event to notify the listener that the modified unit's volume changed
		/// </summary>
		protected void OnUnitVolumeChanged()
		{
			if (UnitVolumeChanged != null)
				UnitVolumeChanged(this, System.EventArgs.Empty);
		}

		/// <summary>
		/// Use this event, which is fired when the bloodunit volume is changed
		/// This bubbles the event and fires up to the control class which will calculate new values
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected void VolumeChanged(object sender, System.EventArgs e)
		{
			bool ruleIsBroken = (this.NewUnit.OriginalVolume > 2000 || this.NewUnit.OriginalVolume < 1);

			if (this.IsSplit)
			{
				this.RuleBroken(Common.VbecsBrokenRules.FrmSplitUnit.Volume, ruleIsBroken);
			}
			else if (this.IsPool)
			{
				this.RuleBroken(Common.VbecsBrokenRules.FrmPoolUnits.Volume, ruleIsBroken);
			}
			else
			{
				this.RuleBroken(Common.VbecsBrokenRules.FrmModifyUnits.Volume, ruleIsBroken);
			}

			this.OnUnitVolumeChanged();
		}

		/// <summary>
		/// Event to capture the modified unit's change
		/// </summary>
		public event EventHandler ValidInvalidChanged;

		/// <summary>
		/// Fires the event to notify the listener that the modified unit's change
		/// </summary>
		protected void OnValidInvalidChanged()
		{
			if (ValidInvalidChanged != null)
				ValidInvalidChanged(this, System.EventArgs.Empty);
		}

		/// <summary>
		/// Use this event, which is fired when the bloodunit is changed
		/// This bubbles the event and fires up to the control class to process broken rules
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected void ChildStatusChange(object sender, System.EventArgs e)
		{
			this.OnValidInvalidChanged();
		}

		#endregion

		#region Properties

		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7487"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7488"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OriginalContainerIndicator
		/// No longer a field on BloodUnit -- UnitContainerLotNumber is all -- and where is it used???
		/// </summary>
		public bool OriginalContainerIndicator
		{
			get
			{
				return(this._originalContainerIndicator);
			}
			set
			{
				this.RuleBroken(M_CONTAINER, (!value && this.Container.LotNumber == string.Empty));
				this._originalContainerIndicator = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="109"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="110"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Unique identifier on the BloodUnitModification table for this unit
		/// </summary>
		public System.Guid BloodUnitModificationGuid
		{
			get
			{
				return (System.Guid)this._modifiedUnitRow[BLOODUNITMODIFICATION.BloodUnitModificationGuid];
			}
			set
			{
				this._modifiedUnitRow[BLOODUNITMODIFICATION.BloodUnitModificationGuid] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="111"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Common.ProductModification</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="112"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Type of modification being performed - ReadOnly, set in constructor.
		/// </summary>
		public Common.ProductModification ProductModificationCode
		{
			get
			{
				return Common.Utility.GetProductModificationCodeFromString(this._modifiedUnitRow[BLOODUNITMODIFICATION.ProductModificationCode].ToString());
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="113"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Common.ModifyUnitMethod</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="114"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Method of performing the modification ... Open/Close, etc. ReadOnly, set in constructor
		/// </summary>
		public Common.ModifyUnitMethod ModificationMethod
		{
			get
			{
				return Common.Utility.GetModificationMethodEnumFromString(this._modifiedUnitRow[BLOODUNITMODIFICATION.ModificationMethodCode].ToString());
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="115"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="116"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Time the current Modification is performed.  ReadOnly, set in constructor.
		/// </summary>
		public System.DateTime ModificationDateTime
		{
			get
			{
				return (System.DateTime)this._modifiedUnitRow[BLOODUNITMODIFICATION.ModificationDate];
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="117"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>BOL.BloodUnit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="118"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This is our essentially our oringal unit which we're going to use since it's there
		/// </summary>
		public BOL.BloodUnit NewUnit
		{
			get
			{
				return this._newBloodUnit;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="123"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="124"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Originanl Bloodunit's Guid
		/// </summary>
		public System.Guid FromBloodUnitGuid
		{
			get
			{
				return (System.Guid)this._modifiedUnitRow[BLOODUNITMODIFICATION.FromBloodUnitGuid];
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="125"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="126"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// New BloodUnitGuid created for the modified unit.  
		/// </summary>
		public System.Guid ToBloodUnitGuid
		{
			get
			{
				return (System.Guid)this._modifiedUnitRow[BLOODUNITMODIFICATION.ToBloodUnitGuid];
			}
			set
			{
				this._modifiedUnitRow[BLOODUNITMODIFICATION.ToBloodUnitGuid] = value;
			}
		}

		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7489"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7490"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// If label verification doesn't pass, the unit is quarantined and patient assignements are not carried over
		/// </summary>
		public bool LabelVerificationFailed
		{
			get
			{
				return (this._labelVerificationFailed);
			}
			set
			{
				this._labelVerificationFailed = value;
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="127"> 
		///		<ExpectedInput>BOL.BloodProduct</ExpectedInput>
		///		<ExpectedOutput>BOL.BloodProduct</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="128"> 
		///		<ExpectedInput>Null</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This should be used to set values against, because the ProductCode property on BloodUnit
		/// monkies with the UnitDivisionId and DonationType -- which we don't need messed with - must set ExpirationDateRule after this!
		/// </summary>
		public BOL.BloodProduct TargetBloodProduct
		{
			get
			{
				return this._targetBloodProduct;
			}
			set
			{
				this._targetBloodProduct = value;

				if (value == null)
				{
					this.RuleBroken(MethodBase.GetCurrentMethod().Name, true);
				}
				else
				{
					this.NewUnit.BloodProductGuid = value.BloodProductGuid;
					this.NewUnit.ProductCode = value.ProductCode;
					this.NewUnit.ProductName = value.ProductName;
					this.NewUnit.ProductShortName = value.ProductShortName;
					this.NewUnit.ProductType = value.ProductType;

					this.RuleBroken(MethodBase.GetCurrentMethod().Name, false);	
				}
				this.OnValidInvalidChanged();
			}

		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="131"> 
		///		<ExpectedInput>DateTime</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="132"> 
		///		<ExpectedInput>DateTime.MinValue</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// NewLabelVerificationDateTime field
		/// </summary>
		public System.DateTime NewLabelVerificationDateTime
		{
			get
			{
				if (this._modifiedUnitRow.IsNull(BLOODUNITMODIFICATION.NewLabelVerificationDatetime))
						return System.DateTime.MinValue;

				return (System.DateTime)this._modifiedUnitRow[BLOODUNITMODIFICATION.NewLabelVerificationDatetime];
			}
			set
			{
				if (value == System.DateTime.MinValue)
				{
					throw new ArgumentException(MethodBase.GetCurrentMethod().Name);
				}
				this._modifiedUnitRow[BLOODUNITMODIFICATION.NewLabelVerificationDatetime] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="256"> 
		///		<ExpectedInput>boolean</ExpectedInput>
		///		<ExpectedOutput>boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="257"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Weld Indicator
		/// </summary>
		public bool WeldIndicator
		{
			get
			{
				return this._weldIndicator;
			}
			set
			{
				this._weldIndicator = value;
			}
		}

		/// <summary>
		/// Returns true if the Modification is any POOL type
		/// </summary>
		public bool IsPool
		{
			get
			{
				return(this.ProductModificationCode == Common.ProductModification.EditPool ||
					this.ProductModificationCode == Common.ProductModification.Pooled ||
					this.ProductModificationCode == Common.ProductModification.PoolThawCryo);
			}
		}


		/// <summary>
		/// Returns true if the BloodUnitModification is a Split
		/// </summary>
		private bool IsSplit
		{
			get
			{
				return(this.ProductModificationCode == Common.ProductModification.SplitAdult);
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="133"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="134"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Container Lot Number for the modified unit
		/// </summary>
		public string UnitContainerLotNumber
		{
			get
			{
				//BR_25.03 - Assign the right container
				if (this._originalContainerIndicator || 
					(this.ModificationMethod == Common.ModifyUnitMethod.Closed && this.ProductModificationCode == Common.ProductModification.SplitAdult))
					return(this._originalContainerLotNumber);
				else
					return(this._container.LotNumber);
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="200"> 
		///		<ExpectedInput>DateTime</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="201"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Not using the BloodUnitMedia property, to save on broken rule collection and bubbling
		/// </summary>
		public System.DateTime UnitExpirationDate
		{
			get
			{
				return this._unitExpirationDateTime;
			}
			set
			{
				this._unitExpirationDateTime = value;

				//We don't have anything to really flag the user on if they 

				RuleBroken(MethodBase.GetCurrentMethod().Name, false);
				//BLERG!  Setting it here as well for bloodunitlabelverification -- but I don't want to use this!!!
				this.NewUnit.BloodUnitMedia.UnitExpirationDate = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="135"> 
		///		<ExpectedInput>ArrayList</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="136"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Equipment List - Arraylist of BOL.Equipment objects attached to the modification
		/// </summary>
		public System.Collections.ArrayList EquipmentList
		{
			get
			{
				return this._equipmentList;
			}
			set
			{
				this._equipmentList = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="137"> 
		///		<ExpectedInput>ArrayList</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="138"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Supply List - array of BOL.Supply objects
		/// </summary>
		public System.Collections.ArrayList SupplyList
		{
			get
			{
				return this._supplyList;
			}
			set
			{
				this._supplyList = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="139"> 
		///		<ExpectedInput>BOL.Supply</ExpectedInput>
		///		<ExpectedOutput>BOL.Supply</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="140"> 
		///		<ExpectedInput>null</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Container
		/// </summary>
		public BOL.Supply Container
		{
			get
			{
				return this._container;
			}
			set
			{
				this._container = value;
				//When we set the container, let's set the LotNumber right away (for BloodUnit record)
				this._newBloodUnit.UnitContainerLotNumber = value.LotNumber;
				this.RuleBroken(M_CONTAINER, (value == null));
				this.OnValidInvalidChanged();
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="141"> 
		///		<ExpectedInput>BOL.Supply</ExpectedInput>
		///		<ExpectedOutput>BOL.Supply</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="142"> 
		///		<ExpectedInput>null</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// SCDWafer
		/// </summary>
		public BOL.Supply SCDWafer
		{
			get
			{
				return this._scdWafer;
			}
			set
			{
				this._scdWafer = value;
				this.RuleBroken(MethodBase.GetCurrentMethod().Name, (value == null));
				this.OnValidInvalidChanged();
			}
		}

		#endregion

		#region Methods


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="143"> 
		///		<ExpectedInput>Form Name</ExpectedInput>
		///		<ExpectedOutput>BrokenRules string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="144"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates a giant broken rules string of bloodunit and producttype
		/// </summary>
		/// <param name="formName">Name of the UI calling form</param>
		/// <returns></returns>
		public new string GetBrokenRulesMessage(string formName)
		{
			System.Text.StringBuilder sb = new System.Text.StringBuilder();

			sb.Append(base.GetBrokenRulesMessage(formName));
			sb.Append(this._newBloodUnit.GetBrokenRulesMessage(formName));

			return sb.ToString();
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="145"> 
		///		<ExpectedInput>FormName, RuleSet strings</ExpectedInput>
		///		<ExpectedOutput>BrokenRuleCount > 0</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="146"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Way to cluster all the broken rules of our bloodunit/ producttype
		/// </summary>
		/// <param name="formName"></param>
		/// <param name="ruleSet"></param>
		public new void SetInitialBrokenRules( string formName, string ruleSet )
		{
			base.SetInitialBrokenRules(formName, ruleSet);
			this._newBloodUnit.SetInitialBrokenRules(formName, "BloodUnit");
			this._targetBloodProduct.SetInitialBrokenRules(formName, "BloodProduct");
		}

		/// <summary>
		/// Fill in  our blank ModifiedUnit class with data
		/// </summary>
		/// <param name="bumControl">ExpirationDate</param>
		private void MakeNew(BOL.BloodUnitModification bumControl)
		{
			//Splits and Pools are actually CREATING new units and need different logic to do so
			if (this.IsPool || this.IsSplit)
			{
				System.Guid newUnitGuid = System.Guid.NewGuid();

				this._modifiedUnitRow[BLOODUNITMODIFICATION.BloodUnitModificationGuid] = System.Guid.NewGuid();
				this._modifiedUnitRow[BLOODUNITMODIFICATION.ToBloodUnitGuid] = newUnitGuid;
				this._modifiedUnitRow[BLOODUNITMODIFICATION.FromBloodUnitGuid] = this._newBloodUnit.BloodUnitGuid;

				//This is a "copy" of the original, so we'll need new GUIDs
				this._newBloodUnit.IncomingShipment.ShipmentGuid = System.Guid.NewGuid();
				this._newBloodUnit.BloodUnitGuid = newUnitGuid;
				this._newBloodUnit.BloodUnitMedia.BloodUnitMediaGuid = System.Guid.NewGuid();
				this._newBloodUnit.BloodUnitStatus.BloodUnitStatusGuid = System.Guid.NewGuid();
				this._newBloodUnit.BloodUnitFinance.BloodUnitFinanceGuid = System.Guid.NewGuid();
			}
			else	//Regular mods are going to overwrite the original units data
			{
				this._modifiedUnitRow[BLOODUNITMODIFICATION.BloodUnitModificationGuid] = System.Guid.NewGuid();
				this._modifiedUnitRow[BLOODUNITMODIFICATION.ToBloodUnitGuid] = this._newBloodUnit.BloodUnitGuid;
				this._modifiedUnitRow[BLOODUNITMODIFICATION.FromBloodUnitGuid] = this._newBloodUnit.BloodUnitGuid;
			}

			//BR_26.39 - shipper should be the facility where the modification is taking place
			BOL.Division div = new BOL.Division(Common.LogonUser.LogonUserDivisionCode);
			//Do this next lines because localsupplier object is diff than guid -- DUMB!!
			this._newBloodUnit.IncomingShipment.LocalSupplier = new BOL.LocalSupplier(div.LocalSupplierGuid, true);
			this._newBloodUnit.IncomingShipment.LocalSupplierGuid = div.LocalSupplierGuid;

			this._equipmentList = new System.Collections.ArrayList();
			this._supplyList = new System.Collections.ArrayList();

			//Start these off as empty so that the save won't include them if they're never assigned values
			this._container = new BOL.Supply(System.Guid.Empty);
			this._scdWafer = new BOL.Supply(System.Guid.Empty);

			//Save off this information of the original unit
			this._originalContainerLotNumber = this._newBloodUnit.UnitContainerLotNumber;
			//BR_26.15 - We don't carry over the lot number from the original unit
			this._newBloodUnit.UnitContainerLotNumber = string.Empty;

			//Save off the original unit's expiration date and calculate the new expirationdate
			this._originalExpirationDateTime = this._newBloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.UnitExpirationDate = this.RecommendedExpirationDateTime(bumControl);

			//Create our starting blood product 
			this._targetBloodProduct = new BOL.BloodProduct(this._newBloodUnit.BloodProductGuid);
		}



		/// <summary>
		/// Attaches the listeners to the bloodunit's valid/invalid and bloodproducts valid/invalid
		/// </summary>
		private void AttachBrokenRulesEvents()
		{
			//Attach our events
			this._newBloodUnit.VolumeChanged += new EventHandler(this.VolumeChanged);

			this.Valid += new EventHandler(this.ChildStatusChange);
			this.Invalid += new EventHandler(this.ChildStatusChange);

			this._newBloodUnit.Valid += new EventHandler(this.ChildStatusChange);
			this._newBloodUnit.Invalid += new EventHandler(this.ChildStatusChange);

			this._targetBloodProduct.Valid += new EventHandler(this.ChildStatusChange);
			this._targetBloodProduct.Invalid += new EventHandler(this.ChildStatusChange);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="147"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="148"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LoadDataRowFromThis
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.BloodUnitModificationGuid))
			{
				dtRow[BLOODUNITMODIFICATION.BloodUnitModificationGuid] = this.BloodUnitModificationGuid;
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.FromBloodUnitGuid))
			{
				dtRow[BLOODUNITMODIFICATION.FromBloodUnitGuid] = this.FromBloodUnitGuid;
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.ModificationDate))
			{
				dtRow[BLOODUNITMODIFICATION.ModificationDate] = this.ModificationDateTime;
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.ModificationMethodCode))
			{
				dtRow[BLOODUNITMODIFICATION.ModificationMethodCode] = Common.Utility.GetModificationMethodCharFromEnum(this.ModificationMethod);
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.NewLabelVerificationDatetime))
			{
				dtRow[BLOODUNITMODIFICATION.NewLabelVerificationDatetime] = this.NewLabelVerificationDateTime;
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.ProductModificationCode))
			{
				dtRow[BLOODUNITMODIFICATION.ProductModificationCode] = Common.Utility.GetProductModificationCharFromEnum(this.ProductModificationCode);
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.ToBloodUnitGuid))
			{
				dtRow[BLOODUNITMODIFICATION.ToBloodUnitGuid] = this.ToBloodUnitGuid;
			}

			return dtRow;
		}


		/// <summary>
		/// LoadFromDataRow
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			       
			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.BloodUnitModificationGuid))
			{
				this._modifiedUnitRow[BLOODUNITMODIFICATION.BloodUnitModificationGuid] = dtRow[BLOODUNITMODIFICATION.BloodUnitModificationGuid];
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.DivisionCode))
			{
				this._modifiedUnitRow[BLOODUNITMODIFICATION.DivisionCode] = dtRow[BLOODUNITMODIFICATION.DivisionCode];
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.FromBloodUnitGuid))
			{
				this._modifiedUnitRow[BLOODUNITMODIFICATION.FromBloodUnitGuid] = dtRow[BLOODUNITMODIFICATION.FromBloodUnitGuid];
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.ModificationDate))
			{
				this._modifiedUnitRow[BLOODUNITMODIFICATION.ModificationDate] = dtRow[BLOODUNITMODIFICATION.ModificationDate];
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.ModificationMethodCode))
			{
				this._modifiedUnitRow[BLOODUNITMODIFICATION.ModificationMethodCode] = dtRow[BLOODUNITMODIFICATION.ModificationMethodCode];
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.ModificationTechId))
			{
				this._modifiedUnitRow[BLOODUNITMODIFICATION.ModificationTechId] = dtRow[BLOODUNITMODIFICATION.ModificationTechId];
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.NewContainerWeldIndicator))
			{
				this._modifiedUnitRow[BLOODUNITMODIFICATION.NewContainerWeldIndicator] = dtRow[BLOODUNITMODIFICATION.NewContainerWeldIndicator];
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.NewLabelVerificationDatetime))
			{
				this._modifiedUnitRow[BLOODUNITMODIFICATION.NewLabelVerificationDatetime] = dtRow[BLOODUNITMODIFICATION.NewLabelVerificationDatetime];
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.ProductModificationCode))
			{
				this._modifiedUnitRow[BLOODUNITMODIFICATION.ProductModificationCode] = dtRow[BLOODUNITMODIFICATION.ProductModificationCode];
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITMODIFICATION.ToBloodUnitGuid))
			{
				this._modifiedUnitRow[BLOODUNITMODIFICATION.ToBloodUnitGuid] = dtRow[BLOODUNITMODIFICATION.ToBloodUnitGuid];
			}

			dtRow = null;
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3706"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>Populated DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3707"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Populates the incoming datarow populated with all relevant Shipment information
		/// </summary>
		/// <param name="dtRow"></param>
		/// <param name="incomingShipmentGuid"></param>
		/// <returns>Datarow with shipment information</returns>
		public System.Data.DataRow LoadBloodUnitShipmentInformation(System.Data.DataRow dtRow, System.Guid incomingShipmentGuid)
		{
			if (dtRow.Table.Columns.Contains(BLOODUNITSHIPMENT.BloodUnitShipmentGuid))
				dtRow[BLOODUNITSHIPMENT.BloodUnitShipmentGuid] = System.Guid.NewGuid();

			if(dtRow.Table.Columns.Contains(BLOODUNITSHIPMENT.BloodUnitGuid))
				dtRow[BLOODUNITSHIPMENT.BloodUnitGuid] = this.ToBloodUnitGuid;

			if(dtRow.Table.Columns.Contains(BLOODUNITSHIPMENT.IncomingShipmentGuid))
				dtRow[BLOODUNITSHIPMENT.IncomingShipmentGuid] = incomingShipmentGuid;

			if(dtRow.Table.Columns.Contains(BLOODUNITSHIPMENT.LastUpdateUser))
				dtRow[BLOODUNITSHIPMENT.LastUpdateUser] = Common.LogonUser.LogonUserName;

			//For Modifed Shipments -- the "M" indicator (RecordStatusCode) is on the SHipment record
			if(dtRow.Table.Columns.Contains(BLOODUNITSHIPMENT.RecordStatusCode))
			{
				dtRow[BLOODUNITSHIPMENT.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			}
			return dtRow;
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="151"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>Populated DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="152"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Populates the input datarow with information specific to the BloodUnit table that will
		/// be used to create the new bloodunit in the database for the new modified unit -- Only used by splits/pool
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns>Datarow with BloodUnitInformation</returns>
		public System.Data.DataRow LoadBloodUnitInformation(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.TableName == BLOODUNIT.TableName)		//Load the BloodUnit info
			{
				dtRow = this._newBloodUnit.LoadDataRowFromThis(dtRow);

				if (dtRow.Table.Columns.Contains(BLOODUNIT.BloodProductGuid))
					dtRow[BLOODUNIT.BloodProductGuid] = this._targetBloodProduct.BloodProductGuid;

				if (dtRow.Table.Columns.Contains(BLOODUNIT.UnitContainerLotNumber))
				{
					if (this.UnitContainerLotNumber.Trim() != string.Empty)
						dtRow[BLOODUNIT.UnitContainerLotNumber] = this.UnitContainerLotNumber;
				}

				//BR_26.22 - Modified unit's InDate is the time of the modification (Creation Date = InDate?)
				if (dtRow.Table.Columns.Contains(BLOODUNIT.UnitInDate))
					dtRow[BLOODUNIT.UnitInDate] = this.ModificationDateTime;

				if (dtRow.Table.Columns.Contains(BLOODUNIT.UnitProductCode))
					dtRow[BLOODUNIT.UnitProductCode] = this._targetBloodProduct.ProductCode;
			}

			if (dtRow.Table.TableName == BLOODUNITFINANCE.TableName)	//Load the BloodUnitFinance info
			{
				dtRow = this._newBloodUnit.BloodUnitFinance.LoadDataRowFromThis(dtRow);
			}

			if (dtRow.Table.TableName == BLOODUNITMEDIA.TableName)		//Load the BloodUnitMedia info
			{
				dtRow = this._newBloodUnit.BloodUnitMedia.LoadDataRowFromThis(dtRow);	

				if (dtRow.Table.Columns.Contains(BLOODUNITMEDIA.UnitExpirationDate))
					dtRow[BLOODUNITMEDIA.UnitExpirationDate] = this.UnitExpirationDate;
			}

			if (dtRow.Table.TableName == BLOODUNITSTATUS.TableName)		//Load the BloodUnitStatus info
			{
				dtRow = this._newBloodUnit.BloodUnitStatus.LoadDataRowFromThis(dtRow);			

				if (dtRow.Table.Columns.Contains(BLOODUNITSTATUS.QuarantineIndicator))
				{
					if (this.LabelVerificationFailed)
						dtRow[BLOODUNITSTATUS.QuarantineIndicator] = true;
					else
						dtRow[BLOODUNITSTATUS.QuarantineIndicator] = this._newBloodUnit.BloodUnitStatus.QuarantineIndicator;
				}

				if (dtRow.Table.Columns.Contains(BLOODUNITSTATUS.UnitStatusDate))
					dtRow[BLOODUNITSTATUS.UnitStatusDate] = this.ModificationDateTime;
			}


			//All of the Above tables should have BloodUnitGuid -- and we need to set it to be our NEW guid, else INDEX voilation
			if (dtRow.Table.Columns.Contains(BLOODUNIT.BloodUnitGuid))
				dtRow[BLOODUNIT.BloodUnitGuid] = this.ToBloodUnitGuid;


			return dtRow;
		}		
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="153"> 
		///		<ExpectedInput>Arraylist of Supplies</ExpectedInput>
		///		<ExpectedOutput>Populated DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="154"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Cycles through the SupplyList array and populates a datatable if needed
		/// </summary>
		/// <param name="bumGuids">Guids of the BloodUnitModification records created - needs cuz of Pooling</param>
		/// <param name="isUpdate"></param>
		/// <returns>DataTable with the Modification Supply information</returns>
		public System.Data.DataTable LoadModificationSupplyInformation(System.Collections.ArrayList bumGuids, bool isUpdate)
		{
			System.Data.DataTable dt = DAL.BloodUnitModification.GetEmptyModificationSupplyTable(isUpdate);

			foreach(System.Guid bumGuid in bumGuids)
			{
				foreach(BOL.Supply tmpSupply in this.SupplyList)
				{
					dt.Rows.Add(this.BuildSupplyRecord(dt, tmpSupply.SupplyGuid, bumGuid, isUpdate));
				}

				//Save any container they may have used in the mod
				if (this.Container.SupplyGuid != System.Guid.Empty)
				{
					dt.Rows.Add(this.BuildSupplyRecord(dt, this.Container.SupplyGuid, bumGuid,isUpdate));
				}

				//Save any SCDWafer they may have used in the mod
				if (this.SCDWafer.SupplyGuid != System.Guid.Empty)
				{
					dt.Rows.Add(this.BuildSupplyRecord(dt, this.SCDWafer.SupplyGuid, bumGuid,isUpdate));
				}
			}
			return dt;
		}

		/// <summary>
		/// Getting some code reuse.  This is called 3 different times in the above method
		/// </summary>
		/// <param name="dt"></param>
		/// <param name="supplyGuid"></param>
		/// <param name="bumGuid">BloodUnitModificationGuid of parent record - had to be done cuz of Pooling</param>
		/// <param name="isUpdate"></param>
		/// <returns></returns>
		private System.Data.DataRow BuildSupplyRecord(System.Data.DataTable dt, System.Guid supplyGuid, System.Guid bumGuid, bool isUpdate)
		{
			System.Data.DataRow dtRow = dt.NewRow();
			
			if (dtRow.Table.Columns.Contains(MODIFICATIONSUPPLY.BloodUnitModificationGuid))
				dtRow[MODIFICATIONSUPPLY.BloodUnitModificationGuid] = bumGuid;

			if (dtRow.Table.Columns.Contains(MODIFICATIONSUPPLY.SupplyGuid))
				dtRow[MODIFICATIONSUPPLY.SupplyGuid] = supplyGuid;
				
			if (dtRow.Table.Columns.Contains(MODIFICATIONSUPPLY.ModificationSupplyGuid))
				dtRow[MODIFICATIONSUPPLY.ModificationSupplyGuid] = System.Guid.NewGuid();
				
			if (dtRow.Table.Columns.Contains(MODIFICATIONSUPPLY.RecordStatusCode))
			{
				if (isUpdate)
				{
					dtRow[MODIFICATIONSUPPLY.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
				}
				else
				{
					dtRow[MODIFICATIONSUPPLY.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
				}
			}

			return(dtRow);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="155"> 
		///		<ExpectedInput>Arraylist of Equipment</ExpectedInput>
		///		<ExpectedOutput>Populated DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="156"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Cycles through the EquipmentList array and populates a datatable if needed
		/// </summary>
		/// <param name="bumGuids">Guids of the BloodUnitModification records created -- had to be done cuz of Pooling</param>
		/// <param name="isUpdate"></param>
		/// <returns>DataTable with the Modification Equipment information</returns>
		public System.Data.DataTable LoadModificationEquipmentInformation(System.Collections.ArrayList bumGuids, bool isUpdate)
		{
			System.Data.DataTable dt = DAL.BloodUnitModification.GetEmptyModificationEquipmentTable(isUpdate);

			foreach(System.Guid bumGuid in bumGuids)
			{
				foreach(BOL.Equipment tmpEquip in this.EquipmentList)
				{
					System.Data.DataRow dtRow = dt.NewRow();

					if (dtRow.Table.Columns.Contains(MODIFICATIONEQUIPMENT.BloodUnitModificationGuid))
						dtRow[MODIFICATIONEQUIPMENT.BloodUnitModificationGuid] = bumGuid;

					if (dtRow.Table.Columns.Contains(MODIFICATIONEQUIPMENT.EquipmentGuid))
						dtRow[MODIFICATIONEQUIPMENT.EquipmentGuid] = tmpEquip.EquipmentGuid;
				
					if (dtRow.Table.Columns.Contains(MODIFICATIONEQUIPMENT.ModificationEquipmentGuid))
						dtRow[MODIFICATIONEQUIPMENT.ModificationEquipmentGuid] = System.Guid.NewGuid();

					if (dtRow.Table.Columns.Contains(MODIFICATIONEQUIPMENT.RecordStatusCode))
					{
						if (isUpdate)
						{
							dtRow[MODIFICATIONEQUIPMENT.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
						}
						else
						{
							dtRow[MODIFICATIONEQUIPMENT.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
						}
					}

					dt.Rows.Add(dtRow);
				}
			}

			return dt;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="157"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="158"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Uses TT26.02 to calculate the expiration date for a modification
		/// Follows BR_26.34 - Adding Days sets hour to 23:59, else follow hour of modification
		/// BR_26.36, BR_24.10
		/// </summary>
		/// <returns></returns>
		public System.DateTime RecommendedExpirationDateTime(BOL.BloodUnitModification bumControl)
		{
			bool isSplit = false;

			System.DateTime targetExpirationDate = System.DateTime.MinValue;
			Common.ModifyUnitMethod modificationMethod = this.ModificationMethod;
				
			//BR_26.30 - If the modificationMethod is SCD, and the weld indicator is false, it's the same as Open
			if (this.ModificationMethod == Common.ModifyUnitMethod.SterileConnectingDevice)
			{
				if (!this.WeldIndicator)
					modificationMethod = Common.ModifyUnitMethod.Open;
			}

			//CR2735 - Returns [closed, open]
			System.Collections.ArrayList expirationRules = bumControl.GetExpirationDateRuleValues(this.TargetBloodProduct);
			//CR2002 - Using Open/Closed Expiration int values from BloodUnitModification object now ( variables are now public access)
			int hoursToAdd = (modificationMethod == Common.ModifyUnitMethod.Open) ? (int)expirationRules[1] : (int)expirationRules[0];

			//Splits get special treatment
			if (this.IsSplit)
			{
				isSplit = true;

				if (modificationMethod == Common.ModifyUnitMethod.Open)
				{
					//BR_25.19 - For OPEN Splits, the expiration date is 24 hours ahead of modification time,
					targetExpirationDate = this.ModificationDateTime.AddHours(24);
				}
				else
				{
					//BR_25.18 - Closed/SCD splits, the expiration date is same as original units
					targetExpirationDate = this._originalExpirationDateTime;
				}
			}
			else
			{
				//Anything greater than 1 day, and we set the time to 23:59 - per BR_26.34
				if (hoursToAdd > 24)
				{
					//Figure out how many days there are
					int daysToAdd = hoursToAdd / 24;

					//To get the time to be 23:59, we have to add the Days,Hours,Minutes to the DATE Property
					targetExpirationDate = this.ModificationDateTime.Date.AddDays(daysToAdd).AddHours(23).AddMinutes(59).AddSeconds(59);
				}
				else
				{
					targetExpirationDate = this.ModificationDateTime.AddHours(hoursToAdd);
				}
			}

			//BR_26.20 -- Always return the lesser, unless it's Rejuv or Freeze
			if (this.ProductModificationCode == Common.ProductModification.Rejuvenated ||
				this.ProductModificationCode == Common.ProductModification.Frozen)
			{
				return(targetExpirationDate);
			}
			else
			{
				//Pools have more than one "original" unit expiration date, so compare them all vs. the system recommended
				if (bumControl.IsPool)
				{
					foreach(BOL.BloodUnit testUnit in bumControl.PoolingUnits)
					{	
						//BR_24.13 -- we want to go with the smallest expiration date possible
						if (testUnit.BloodUnitMedia.UnitExpirationDate < targetExpirationDate)
							targetExpirationDate = testUnit.BloodUnitMedia.UnitExpirationDate;
					}
					return(targetExpirationDate);
				}
				else
				{
					//When hours = 0, this implies to use the expiration date of the unit
					if (this._originalExpirationDateTime < targetExpirationDate || (hoursToAdd == 0 && !isSplit))
						return(this._originalExpirationDateTime);
					else
						return(targetExpirationDate);
				}
			}
		}


		#endregion


	}
}
